# -*- coding: cp1252 -*-
#
#	FILE:	 BezierContinents.py
#	AUTHOR:  Toni Leppkorpi
#	PURPOSE: Global map script - Bezier generated continents
#-----------------------------------------------------------------------------
#	Copyleft (c) 2005
#-----------------------------------------------------------------------------
#

import math
if __name__ != '__main__':
    from CvPythonExtensions import *
    import CvMapGeneratorUtil
    from CvMapGeneratorUtil import TerrainGenerator
    from CvMapGeneratorUtil import FeatureGenerator
else:
    #dummy code for fast command line testing
    import sys
    import random
    class CyGlobalContext:
        def getGame(self):
            return Game()

    class Game:
        def getMapRand(self):
            return Dice()
    
    class Dice:
        def get(self, n, msg):
            if n == 1:
                return 0
            return random.randint(0,n-1)

    class CyPythonMgr:
        def debugMsg(self, msg):
            print msg

    class CyFractal:
        def fracInit(self, w, h, grain, dice, flag, xe, ye):
            self.grain = grain
        def getHeightFromPercent(self, p):
            return p
        def getHeight(self, x, y):
            return random.randint(0,99)

    class CyMap:
        def getGridWidth(self):
            return 79
        def getGridHeight(self):
            return 50
        def getCustomMapOption(self, n):
            if len(sys.argv) > n+1:
                return int(sys.argv[n+1])
            else:
                return getCustomMapOptionDefault([n])
        def getSeaLevel(self):
            n = getNumCustomMapOptions()+1
            if len(sys.argv) > n:
                return int(sys.argv[n])
            else:
                return 1

    def NiTextOut(txt):
        print txt

    class PlotTypes:
        NO_PLOT = -1
        PLOT_PEAK = 0
        PLOT_HILLS = 1
        PLOT_LAND = 2
        PLOT_OCEAN = 3
        NUM_PLOT_TYPES = 4
    #dummy code ends


def getDescription():
	return "TXT_KEY_MAP_SCRIPT_BEZIER_DESCR"
	
def isAdvancedMap():
	"This map should show up in simple mode"
	return 0

class MapType:
    CONTINENTS = 0
    PANGAEA = 1
    MIXED = 2
    DEFAULT = MIXED

class ShapeType:
    NATURAL = 0
    PRESSED = 1
    SOLID = 2
    DEFAULT = NATURAL
	
class MountainType:
    IMPASSABLE = 0
    DENSE = 1
    NORMAL = 2
    BRANCHED = 3
    DEFAULT = NORMAL

askMapType = True
askMountainPattern = True
askLandShape = False

baseLandPercent = 32
hillsPercent = 30
branchPercent = 5
archipelago = True
debug = False

def getOptionString():
    s = ''
    if askMapType: s += '[MapType 0-2] '
    if askMountainPattern: s += '[MountainPattern 0-3] '
    if askLandShape: s += '[LandShape 0-2] '
    s += '[SeaLevel 0-2]'
    return s

def getNumCustomMapOptions():
    n = 0
    if askMapType: n += 1
    if askMountainPattern: n += 1
    if askLandShape: n += 1
    return n
	
def getCustomMapOptionName(argsList):
    [iOption] = argsList
    optionNames = []
    if askMapType: optionNames.append("TXT_KEY_MAP_SCRIPT_BEZIER_TYPE")
    if askMountainPattern: optionNames.append("TXT_KEY_MAP_SCRIPT_MOUNTAIN_PATTERN")
    if askLandShape: optionNames.append("TXT_KEY_MAP_SCRIPT_LAND_SHAPE")
    translated_text = unicode(CyTranslator().getText(optionNames[iOption], ()))
    return translated_text
	
def getNumCustomMapOptionValues(argsList):
    [iOption] = argsList
    optionValues = []
    if askMapType: optionValues.append(3)
    if askMountainPattern: optionValues.append(4)
    if askLandShape: optionValues.append(3)
    return optionValues[iOption]
	
def getCustomMapOptionDescAt(argsList):
    [iOption, iSelection] = argsList
    selectionNames = []
    if askMapType:
        selectionNames.append([
            "TXT_KEY_MAP_SCRIPT_BEZIER_CONTINENTS",
            "TXT_KEY_MAP_SCRIPT_BEZIER_PANGAEA",
            "TXT_KEY_MAP_SCRIPT_BEZIER_MIXED",
            ])
    if askMountainPattern:
        selectionNames.append([
            "TXT_KEY_MAP_SCRIPT_BEZIER_IMPASSABLE",
            "TXT_KEY_MAP_SCRIPT_DENSE_PEAKS",
            "TXT_KEY_MAP_SCRIPT_NORMAL_PEAKS",
            "TXT_KEY_MAP_SCRIPT_BEZIER_BRANCHED",
            ])
    if askLandShape:
        selectionNames.append([
            "TXT_KEY_MAP_SCRIPT_NATURAL",
            "TXT_KEY_MAP_SCRIPT_PRESSED",
            "TXT_KEY_MAP_SCRIPT_SOLID"
            ])
    translated_text = unicode(CyTranslator().getText(selectionNames[iOption][iSelection], ()))
    return translated_text

def getCustomMapOptionDefault(argsList):
    [iOption] = argsList
    optionDefaults = []
    if askMapType: optionDefaults.append(MapType.DEFAULT)
    if askMountainPattern: optionDefaults.append(MountainType.DEFAULT)
    if askLandShape: optionDefaults.append(ShapeType.DEFAULT)
    return optionDefaults[iOption]

def isClimateMap():
	return 1

def isSeaLevelMap():
	return 1

def getWrapX():
	return True
def getWrapY():
	return False

class Bezier:
    MAXITER = 12
    def __init__(self, p1, p2, p3, p4):
	self.dice = CyGlobalContext().getGame().getMapRand()
        self.p1x = p1[0]-0.5
        self.p1y = p1[1]-0.5
        self.p2x = p2[0]-0.5
        self.p2y = p2[1]-0.5
        self.p3x = p3[0]-0.5
        self.p3y = p3[1]-0.5
        self.p4x = p4[0]-0.5
        self.p4y = p4[1]-0.5

    def calculate(self, t):
        x = self.p1x*(1-t)**3 + self.p2x*3.0*(1-t)**2*t + self.p3x*3.0*(1-t)*t**2 + self.p4x*t**3
        y = self.p1y*(1-t)**3 + self.p2y*3.0*(1-t)**2*t + self.p3y*3.0*(1-t)*t**2 + self.p4y*t**3
        return (int(x),int(y))

    def draw(self, continent):
        p_first = (int(self.p1x), int(self.p1y))
        p_last  = (int(self.p4x), int(self.p4y))
        continent.setPoint(p_first, Continent.ENDPOINT)
        continent.setPoint(p_last, Continent.ENDPOINT)
        self.iterDraw(continent, p_first, 0.0, p_last, 1.0, 0)

    def iterDraw(self, continent, p1, t1, p2, t2, i):
        dx = abs(p1[0]-p2[0])
        dy = abs(p1[1]-p2[1])
        #if max(dx,dy) <= 1:
        #    return
        if dx+dy <= 1:
            return
        t_mid = (t1+t2)/2.0
        p_mid = self.calculate(t_mid)
        continent.setPoint(p_mid, Continent.MIDPOINT)
        if i < self.MAXITER:
            self.iterDraw(continent, p1, t1, p_mid, t_mid, i+1)
            self.iterDraw(continent, p_mid, t_mid, p2, t2, i+1)
        elif i == self.MAXITER: #we want impassable mountain ridge
            if self.dice.get(2, "BezierContinents PYTHON") == 0:
                continent.setPoint((p1[0],p2[1]), Continent.MIDPOINT)
            else:
                continent.setPoint((p2[0],p1[1]), Continent.MIDPOINT)
            

class Continent:
    MIDPOINT = 2**29
    ENDPOINT = int(0.75*MIDPOINT)
    PEAK =  MIDPOINT
    HILLS = int(0.6*MIDPOINT)
    MAYBEHILLS = MIDPOINT / 350
    ISLAND = 1000
    ISLANDHILLS = 1001
    COAST = 2
    NEARCOAST = 1
    
    def __init__(self, w, h):
        self.w = w
        self.h = h
        self.initData()
	self.dice = CyGlobalContext().getGame().getMapRand()

    def initData(self):
        self.data = [None]*self.h
        for y in range(self.h):
            self.data[y] = [0]*self.w

    def setPoint(self, p, v):
        self.data[p[1]%self.h][p[0]%self.w] = v

    def isPeak(self, x, y):
        if self.data[y][x] >= self.PEAK:
            return True
        else:
            return False

    def isHills(self, x, y):
        if self.data[y][x] >= self.HILLS or self.data[y][x] == self.ISLANDHILLS:
            return True
        else:
            return False

    def isIsland(self, x, y):
        if self.data[y][x] == self.ISLAND or self.data[y][x] == self.ISLANDHILLS:
            return True
        else:
            return False

    def isOcean(self, x, y):
        if self.data[y][x] <= self.COAST:
            return True
        else:
            return False

    def drawSpine(self, size):
        minX = (self.w-size)/2
        minY = (self.h-size)/2
        p1 = (minX + self.dice.get(size, "BezierContinents PYTHON"), minY + self.dice.get(size, "BezierContinents PYTHON"))
        p2 = (minX + self.dice.get(size, "BezierContinents PYTHON"), minY + self.dice.get(size, "BezierContinents PYTHON"))
        p3 = (minX + self.dice.get(size, "BezierContinents PYTHON"), minY + self.dice.get(size, "BezierContinents PYTHON"))
        p4 = (minX + self.dice.get(size, "BezierContinents PYTHON"), minY + self.dice.get(size, "BezierContinents PYTHON"))
        bez = Bezier(p1, p2, p3, p4)
        bez.draw(self)

    def addMountainBranches(self, branchPercent):
        def peakTest(x, y):
            if self.data[y][x] >= self.PEAK:
                return True
            else:
                return False
        branchLength = 5
        peakMap = self.makeDistanceMap(branchLength, peakTest)
        for y in range(1,self.h-1):
            for x in range(0,self.w):
                if peakMap[y][x] == branchLength and self.dice.get(100, "BezierContinents PYTHON") < branchPercent:
                    self.addOneBranch(x, y, peakMap, branchLength)

    def addOneBranch(self, x, y, peakMap, d):
        self.data[y][x] = self.ENDPOINT
        while d > 1:
            d = d - 1
            pList = []
            if y > 1:
                if peakMap[y-1][(x-1)%self.w] == d: pList.append((y-1,(x-1)%self.w))
                if peakMap[y-1][ x          ] == d: pList.append((y-1, x          ))
                if peakMap[y-1][(x+1)%self.w] == d: pList.append((y-1,(x+1)%self.w))
            if True:
                if peakMap[y  ][(x-1)%self.w] == d: pList.append((y  ,(x-1)%self.w))
                if peakMap[y  ][(x+1)%self.w] == d: pList.append((y  ,(x+1)%self.w))
            if y < self.h-2:
                if peakMap[y+1][(x-1)%self.w] == d: pList.append((y+1,(x-1)%self.w))
                if peakMap[y+1][ x          ] == d: pList.append((y+1, x          ))
                if peakMap[y+1][(x+1)%self.w] == d: pList.append((y+1,(x+1)%self.w))
            n = len(pList)
            if n > 1:
                j = self.dice.get(n, "BezierContinents PYTHON")
            else:
                j = 0
            y,x = pList[j]
            self.data[y][x] = self.MIDPOINT

    def expandSpine(self, n):
        for i in range(n):
            tmp = self.data
            self.initData()
            for y in range(1,self.h-1):
                for x in range(0,self.w):
                    v = tmp[y-1][x] + tmp[y+1][x] + tmp[y][(x-1)%self.w] + tmp[y][(x+1)%self.w]
                    v = v/4
                    self.data[y][x] = max(tmp[y][x], v)
                    
    def expandCoastline(self, n):
        wall = [None]*self.h
        for y in range(self.h):
            wall[y] = [None]*self.w
            for x in range(self.w):
                wall[y][x] = self.dice.get(15, "BezierContinents PYTHON")
        for i in range(n):
            tmp = self.data
            self.initData()
            for y in range(1,self.h-1):
                for x in range(0,self.w):
                    v = 0
                    if wall[y][x] & 1 == 1: v += tmp[y-1][x]
                    if wall[y][x] & 2 == 2: v += tmp[y+1][x]
                    if wall[y][x] & 4 == 4: v += tmp[y][(x-1)%self.w]
                    if wall[y][x] & 8 == 8: v += tmp[y][(x+1)%self.w]
                    v = v/(3+self.dice.get(4, "BezierContinents PYTHON"))
                    self.data[y][x] = max(tmp[y][x], v)

    def makeDistanceMap(self,maxDist,testFunc):
        distMap = [None]*self.h
        for y in range(self.h):
            distMap[y] = [9999]*self.w
        for y in range(0,self.h):
            for x in range(0,self.w):
                if testFunc(x,y):
                    distMap[y][x] = 0
        for d in range(0, maxDist):
            for y in range(0,self.h):
                for x in range(0,self.w):
                    v = 9999
                    if y > 0:
                        v = min(v, distMap[y-1][(x-1)%self.w], distMap[y-1][x], distMap[y-1][(x+1)%self.w])
                    if True:
                        v = min(v, distMap[y  ][(x-1)%self.w],                  distMap[y  ][(x+1)%self.w])
                    if y < self.h-1:
                        v = min(v, distMap[y+1][(x-1)%self.w], distMap[y+1][x], distMap[y+1][(x+1)%self.w])
                    if distMap[y][x] == 9999 and v == d:
                        distMap[y][x] = d+1
        return distMap

    def markCoast(self):
        for y in range(1,self.h-1):
            for x in range(0,self.w):
                if self.data[y][x] <= self.COAST:
                    self.data[y][x] = 0
        def landTest(x, y):
            if self.data[y][x] > self.COAST:
                return True
            else:
                return False
        coastMap = self.makeDistanceMap(2, landTest)
        for y in range(0,self.h):
            for x in range(0,self.w):
                if coastMap[y][x] == 1:
                    self.data[y][x] = self.COAST
                elif coastMap[y][x] == 2:
                    self.data[y][x] = self.NEARCOAST

    def makeHills(self, hillsPercent):
	hillsFrac = CyFractal()
	hillsFrac.fracInit(self.w, self.h, 5, self.dice, 0, -1, -1)
	hillsLowThreshold = hillsFrac.getHeightFromPercent(40)
	hillsHighThreshold = hillsFrac.getHeightFromPercent(40+hillsPercent)
        for y in range(self.h):
            for x in range(self.w):
                p = self.data[y][x]
                val = hillsFrac.getHeight(x,y)
                if hillsLowThreshold <= val < hillsHighThreshold:
                    if p >= self.MAYBEHILLS and p < self.HILLS:
                        self.data[y][x] = self.HILLS
                    elif p == self.ISLAND:
                        self.data[y][x] = self.ISLANDHILLS

    def collapsePeaks(self, collapsePercent):
        if collapsePercent == 0:
            return
        for y in range(self.h):
            for x in range(self.w):
                if self.isPeak(x,y):
                    if self.dice.get(100, "BezierContinents PYTHON") < collapsePercent:
                        self.data[y][x] = self.HILLS

    def getIntersectionNumbers(self, other, dx, dy):
        nI = 0
        total = 0
        maxI = 0
        for y in range(0,self.h):
            for x in range(0,self.w):
                if other.data[y][x] > 0:
                    v = self.data[(y+dy)%self.h][(x+dx)%self.w]
                    if other.data[y][x] > self.COAST:
                        total += 1
                        if v > self.COAST:
                            nI += 1
                    if v > 0:
                        i = other.data[y][x] + v
                        if i > maxI: maxI = i
        return float(nI)/total, maxI

    def doUnion(self, other, dx, dy):
        for y in range(0,self.h):
            for x in range(0,self.w):
                v1 = self.data[y][x]
                v2 = other.data[(y-dy)%self.h][(x-dx)%self.w]
                if v1 == 0: self.data[y][x] = v2
                elif v2 == 0: self.data[y][x] = v1
                else: self.data[y][x] = max(v1, v2)
        self.evaluate()

    def tryUnion(self, other, typeInput):
        if typeInput == MapType.CONTINENTS:
            #Need continents separated by atleast two ocean square
            lowLimit = 0
            highLimit = 0
            pLimit = 0.0
        elif typeInput == MapType.PANGAEA:
            #Need pandaea, collision required
            lowLimit = 2*self.ENDPOINT
            highLimit = 2*self.MIDPOINT
            pLimit = 0.3
        else: #MapType.MIXED
            #Atmost separated by one ocean square, atmost 5% collision
            lowLimit = 2*self.NEARCOAST
            highLimit = self.MIDPOINT
            pLimit = 0.05

        for loop in range(0,32):
            dx = self.dice.get(self.w, "BezierContinents PYTHON")
            if other.maxY-other.minY < self.h-3:
                dy = -(other.minY-1) + self.dice.get(self.h-2-(other.maxY-other.minY), "BezierContinents PYTHON")
            else:
                dy = 0
            p, i = self.getIntersectionNumbers(other, dx, dy)
            if lowLimit <= i <= highLimit and p <= pLimit:
                self.doUnion(other, dx, dy)
                self.evaluate()
                self.markCoast()
                if debug:
                    CyPythonMgr().debugMsg('BezierContinents union operation:')
                    CyPythonMgr().debugMsg('i=%d p=%f dx=%d dy=%d'%(i,p,dx,dy))
                    self.write()
                return True
        return False

    def doBestSplitX(self):
        bestX = -1
        bestV = 9999999999
        for x in range(0,self.w):
            v = self.dice.get(100, "BezierContinents PYTHON")
            for y in range(0,self.h):
                if self.data[y][x] > self.COAST:
                    v += 100000
                else:
                    v += 100*self.data[y][x]
            if v < bestV:
                bestV = v
                bestX = x
        tmp = self.data
        self.initData()
        for y in range(0,self.h):
            for x in range(0,self.w):
                self.data[y][x] = tmp[y][(x+bestX)%self.w]

    def evaluate(self):
        self.nLandOrHills = 0
        self.minX = 99999
        self.maxX = 0
        self.minY = 99999
        self.maxY = 0
        for y in range(0,self.h):
            for x in range(0,self.w):
                if self.NEARCOAST < self.data[y][x]:
                    if self.data[y][x] < self.PEAK:
                        self.nLandOrHills += 1
                    if x < self.minX: self.minX = x
                    if x > self.maxX: self.maxX = x
                    if y < self.minY: self.minY = y
                    if y > self.maxY: self.maxY = y
        return self.nLandOrHills

    def addCoastIslands(self):
	coastFrac = CyFractal()
	coastFrac.fracInit(self.w, self.h, 5, self.dice, 0, -1, -1)
	coastThreshold = coastFrac.getHeightFromPercent(5)
        for y in range(1,self.h-1):
            for x in range(0,self.w):
                if self.data[y][x] == self.NEARCOAST and coastFrac.getHeight(x,y) < coastThreshold:
                    self.createIsland(x,y)

    def createIsland(self, x, y):
        plotList = [(x,y)]
        i = 0
        k = 1
        while i < len(plotList) and k < 20:
            p = plotList[i]
            i += 1
            if k == 1 or self.dice.get(k, "BezierContinents PYTHON") == 0:
                k = k + 1
                x = p[0]
                y = p[1]
                if y in (0, self.h-1):
                    continue
                self.data[y][x] = self.ISLAND
                newList = []
                if self.data[(y-1)%self.h][(x-1)%self.w] <= self.NEARCOAST and ((x-1)%self.w,(y-1)%self.h) not in plotList: newList.append(((x-1)%self.w,(y-1)%self.h))
                if self.data[(y-1)%self.h][ x          ] <= self.NEARCOAST and ( x          ,(y-1)%self.h) not in plotList: newList.append(( x          ,(y-1)%self.h))
                if self.data[(y-1)%self.h][(x+1)%self.w] <= self.NEARCOAST and ((x+1)%self.w,(y-1)%self.h) not in plotList: newList.append(((x+1)%self.w,(y-1)%self.h))
                if self.data[ y          ][(x-1)%self.w] <= self.NEARCOAST and ((x-1)%self.w, y          ) not in plotList: newList.append(((x-1)%self.w, y          ))
                if self.data[ y          ][(x+1)%self.w] <= self.NEARCOAST and ((x+1)%self.w, y          ) not in plotList: newList.append(((x+1)%self.w, y          ))
                if self.data[(y+1)%self.h][(x-1)%self.w] <= self.NEARCOAST and ((x-1)%self.w,(y+1)%self.h) not in plotList: newList.append(((x-1)%self.w,(y+1)%self.h))
                if self.data[(y+1)%self.h][ x          ] <= self.NEARCOAST and ( x          ,(y+1)%self.h) not in plotList: newList.append(( x          ,(y+1)%self.h))
                if self.data[(y+1)%self.h][(x+1)%self.w] <= self.NEARCOAST and ((x+1)%self.w,(y+1)%self.h) not in plotList: newList.append(((x+1)%self.w,(y+1)%self.h))
                n = len(newList)
                while n > 0:
                    if n > 1:
                        j = self.dice.get(n, "BezierContinents PYTHON")
                    else:
                        j = 0
                    plotList.append(newList.pop(j))
                    n = n - 1

    def convertToArchipelago(self):
	archiFrac = CyFractal()
	archiFrac.fracInit(self.w, self.h, 4, self.dice, 0, -1, -1)
	archiThreshold = archiFrac.getHeightFromPercent(40)
        for y in range(0,self.h):
            for x in range(0,self.w):
                if self.data[y][x] > self.COAST:
                    if archiFrac.getHeight(x,y) < archiThreshold:
                        self.data[y][x] = self.ISLAND
                    else:
                        self.data[y][x] = 0

    def write(self):
	mgr = CyPythonMgr()
	mgr.debugMsg('\nBezierContinents map dump:')
        for i in range(len(self.data), 0, -1):
            line = self.data[i-1]
            s = ''
            for plot in line:
                if plot == 0 or (not debug and plot <= self.COAST):
                    s += '.'
                elif plot >= self.PEAK:
                    s += 'X'
                elif plot >= self.HILLS:
                    s += 'h'
                elif plot >= self.MAYBEHILLS:
                    s += '+'
                elif plot == self.ISLAND:
                    s += 'i'
                elif plot == self.ISLANDHILLS:
                    s += 'I'
                elif plot == self.COAST:
                    s += 'c'
                elif plot == self.NEARCOAST:
                    s += 'o'
                else:
                    s += '-'
            mgr.debugMsg(s)
        if debug:
            mgr.debugMsg('size:%d west:%d south:%d east:%d north:%d\n'%(self.nLandOrHills,self.minX, self.minY, self.maxX, self.maxY))

    def getPlotTypes(self):
	plotTypes = [PlotTypes.PLOT_LAND] * (self.w*self.h)
        for y in range(self.h):
            for x in range(self.w):
                i = y*self.w + x
		if self.isOcean(x,y):
                    plotTypes[i] = PlotTypes.PLOT_OCEAN
                elif self.isPeak(x,y):
                    plotTypes[i] = PlotTypes.PLOT_PEAK
                elif self.isHills(x,y):
                    plotTypes[i] = PlotTypes.PLOT_HILLS
                else:
                    plotTypes[i] = PlotTypes.PLOT_LAND
	return plotTypes

def generatePlotTypes():
	NiTextOut("Setting Plot Types (Python BezierContinents) ...")
	gc = CyGlobalContext()
	map = CyMap()
	dice = gc.getGame().getMapRand()
	iW = map.getGridWidth()
	iH = map.getGridHeight()
	
	# Get custom map user inputs.
	typeInput  = map.getCustomMapOption(0)
	shapeInput = 0 #always natural
	mountainInput = map.getCustomMapOption(1)
	seaLevel = map.getSeaLevel()

        landPercent = baseLandPercent-4*(seaLevel-1)

        shapeSizes = (
            ((2,8),(3,7),(6,1)),
            ((2,5),(3,4),(4,1)),
            ((1,4),(2,3),(3,1)),
            )
        spineSize, coastSize = shapeSizes[seaLevel][shapeInput]

        collapsePercent = (0,15,30,20)[mountainInput]
        
        continents = []
        bestMap = None
        continentMaxSize = min(iW, iH) - 6
        bestLandOrHills = 0
        minLandmass = int((landPercent-5)*iW*iH/100)
        maxLandmass = int((landPercent+5)*iW*iH/100)

        while bestLandOrHills < minLandmass and continentMaxSize > 3:
            newContinent = Continent(iW, iH)
            newContinent.drawSpine(continentMaxSize)
            if mountainInput == MountainType.BRANCHED:
                newContinent.addMountainBranches(branchPercent)
            newContinent.expandSpine(spineSize)
            newContinent.expandCoastline(coastSize)
            newContinent.collapsePeaks(collapsePercent)
            newContinent.evaluate()
            newContinent.markCoast()
            if shapeInput == ShapeType.NATURAL:
                newContinent.addCoastIslands()
                newContinent.evaluate()
                newContinent.markCoast()
            if debug:
                newContinent.write()
            if bestMap == None:
                continents = [newContinent]
                bestMap = newContinent
            else:
                for cont in continents:
                    if cont.nLandOrHills + newContinent.nLandOrHills <= maxLandmass:
                        if cont.tryUnion(newContinent, typeInput):
                            if cont.nLandOrHills > bestLandOrHills:
                                bestLandOrHills = cont.nLandOrHills
                                bestMap = cont
                continents.append(newContinent)
            continentMaxSize -= 1

        if archipelago:
            if typeInput == MapType.PANGAEA:
                typeInput = MapType.MIXED
            success = False
            while not success and continentMaxSize > 3:
                newContinent = Continent(iW, iH)
                newContinent.drawSpine(continentMaxSize)
                newContinent.expandSpine(spineSize)
                newContinent.expandCoastline(coastSize)
                newContinent.convertToArchipelago()
                newContinent.evaluate()
                newContinent.markCoast()
                if debug:
                    newContinent.write()
                if bestMap.tryUnion(newContinent, typeInput):
                    success = True
                continentMaxSize -= 1

        bestMap.doBestSplitX()
        global final_map
        final_map = bestMap
        final_map.makeHills(hillsPercent)
        return final_map.getPlotTypes()        

def generateTerrainTypes():
	NiTextOut("Generating Terrain (Python Continents) ...")
	terraingen = TerrainGenerator()
	terrainTypes = terraingen.generateTerrain()
	return terrainTypes

def addFeatures():
	NiTextOut("Adding Features (Python Continents) ...")
	featuregen = FeatureGenerator()
	featuregen.addFeatures()
	return 0

def findStartingPlot(argsList):
	[playerID] = argsList
	def isValid(playerID, x, y, map=final_map):
            if map.isIsland(x,y):
                return False
            else:
                return True
	
        return CvMapGeneratorUtil.findStartingPlot(playerID, isValid)

def normalizeRemovePeaks():
    #dont touch my peaks
    return None
##    map = CyMap()
##    mountainInput = map.getCustomMapOption(1)
##    if mountainInput == MountainType.IMPASSABLE:
##        return None
##    else:
##        CyPythonMgr().allowDefaultImpl()

if __name__ == '__main__':
    if len(sys.argv) == 2 and '?' in sys.argv[1]:
        optionString = getOptionString()
        print 'BezierContinents.py ' + optionString
        sys.exit()
    generatePlotTypes()
    final_map.write()
